/*********************************************************
/                                                        /
/ source code copyright University of Pittsburgh,        /
/ released under the terms of the GPL, see license.txt   /
/                                                        /
/********************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "pdb.h"
#include "vho.h"

#include "res_index.c"
#include "check_modified_aa.c"
#include "generate_hessian.c" 
#include "svd.c"
#include "assign_radius.c"
#include "kirchhof_surfaces1.c"
#include "brendansurfaces.c"

int main (int argc , char *argv[])   {

void print_eigenvalues (pdb *pdbobj , float *d) ;
void print_eigenvectors (pdb *pdbobj , float *d , float **v) ;
void pseudoinverse(float d[], float **v, float **crc, int n) ;
void print_hessian(pdb *pdbobj , float **invhessian) ;
void print_sparse_hessian(pdb *pdbobj , float **hessian , int natoms) ;
void print_pseudoinverse(pdb *pdbobj , float **invhessian) ;
void print_calb(pdb *pdbobj , float **invhessian) ;
void print_indexing(pdb *pdbobj , double mean , double std) ;
void print_log_file(char *file , char chain , int number_of_residues , float th) ; 
void calculate_bfac_stat(pdb *pdbobj , char chain , double *mean , double *std) ;
void print_kirchhof(pdb *pdbobj , float **kirchhof) ;

float **v ;
float *d ;
float **hessian , **invhessian , **cc ,  **modes ;
float **kirmat ;
int nrot ;
int num_of_residues ;
int natoms ;
int n,m ;
pdb *pdbobj ;
char chain ;
double mean, std ;
int cutoff ;  // New addition 12/2005. cutoff is a command line parameter
double ks     ;  // New parameter to determine the ratio between surfaces and coordination

pdbobj = malloc (sizeof(pdb));
chain = argv[2][0] ;
if (chain == '_') chain = ' ' ;
if (chain == '0') chain = ' ' ;

readpdb(argv[1] , pdbobj , chain) ;   // argv[1]: pdb file , argv[2] : chain
cutoff = atof( argv[3]) ;
ks = atof( argv[4]) ;
// printf ("numres is %d\n" , pdbobj->numres) ;
num_of_residues = pdbobj->numres   ;

// memory allocations:

// kirmat = malloc ((12*num_of_residues+1)*sizeof(float *)) ;
hessian = malloc ((36*num_of_residues+3)*sizeof(double *)) ;
// v = malloc ((3*num_of_residues+3)*sizeof(double *)) ;
d = malloc ((3*num_of_residues+3)*sizeof(double)) ;

// for (n=1 ; n<=14*num_of_residues ; n++)  {
//         kirmat[n] = malloc ((14*num_of_residues+1)*sizeof(float));
// }
for (n=1 ; n<=36*num_of_residues ; n++)  {
        // hessian[n] = malloc ((36*num_of_residues+3)*sizeof(double));
	hessian[n] = malloc (5*sizeof(double));
        // v[n]   = malloc ((18*num_of_residues+3)*sizeof(double));
	//for (m=1 ; m<=36*num_of_residues ; m++)  {
	for (m=1 ; m<5 ; m++)  {
		hessian[n][m] = 0.0 ;
	}
}


// create_kirchhof_matrix(pdbobj , kirmat) ;
// print_kirchhof(pdbobj , kirmat) ;
generate_hessian(pdbobj , argv[2][0] , kirmat , hessian , &natoms , cutoff , ks) ;
// print_hessian(pdbobj , hessian , natoms) ;
// print_sparse_hessian(pdbobj , hessian , natoms ) ;
calculate_bfac_stat(pdbobj , chain , &mean , &std) ;
print_indexing(pdbobj , mean , std) ;

exit (1) ;

svdcmp(hessian, 3*num_of_residues , 3*num_of_residues , d , v) ;

v = hessian ; // the svdcmp function  assigns the orthogonal basis to the input pointer (hessian)

eigsrt(d, v, 3*num_of_residues) ;
pseudoinverse(d, v, invhessian , num_of_residues) ;
print_log_file(argv[1] , argv[2][0] , num_of_residues , cutoff) ;
print_eigenvalues(pdbobj , d) ; 
print_eigenvectors(pdbobj , d , v) ; 
print_pseudoinverse(pdbobj , invhessian) ;
print_calb(pdbobj , invhessian) ;

}


void print_eigenvalues (pdb *pdbobj , float *d)  {
	int n ;
	int num_of_residues ;
	FILE *fh ;	

	char file[50] ;
	
	num_of_residues = pdbobj->numres ;
	
	strcpy(file , pdbobj->file) ;
	fh =  fopen(strcat(file, ".eigenvalues") , "w") ;
	for (n=1 ;  n<=3*num_of_residues ; n++)  {
		fprintf (fh , "%10.5f\n" , d[n]) ;
	}
	fclose (fh) ;
}

void print_eigenvectors (pdb *pdbobj , float *d , float **v)  {
	int m,n ;
	int num_of_residues ;
	FILE *fh , *fx , *fy , *fz ;
	char file[50] ;
	int filenamelength ;
	
	num_of_residues = pdbobj->numres ;
	
	strcpy(file , pdbobj->file) ;
	filenamelength = strlen(file) ;
	
	fh =  fopen(strcat(file, ".eigen") , "w") ;
	for (n=1 ;  n<=3*num_of_residues ; n++)  {
		for (m=1 ;  m<=3*num_of_residues ; m++)  {
			fprintf (fh , "%10.5f " , v[n][m]) ;
		}
		fprintf (fh , "\n") ;
	}
	fclose (fh) ;
	
	file[filenamelength] = '\0' ;
	fx =  fopen(strcat(file, ".eigenx") , "w") ;
	file[filenamelength] = '\0' ;
	fy =  fopen(strcat(file, ".eigeny") , "w") ;
	file[filenamelength] = '\0' ;
	fz =  fopen(strcat(file, ".eigenz") , "w") ;
	for (n=1 ;  n<=num_of_residues ; n++)  {
		for (m=7 ;  m<=3*num_of_residues ; m++)  {
			fprintf (fx , "%10.5f " , v[n*3-2][m]) ;
			fprintf (fy , "%10.5f " , v[n*3-1][m]) ;
			fprintf (fz , "%10.5f " , v[n*3][m]) ;
		}
		fprintf (fx , "\n") ;
		fprintf (fy , "\n") ;
		fprintf (fz , "\n") ;
	}
	fclose (fx) ;
	fclose (fy) ;
	fclose (fz) ;
	
	
}

void print_hessian(pdb *pdbobj , float **hessian)  {
int m,n ;
int num_of_residues ;
FILE *fh ;	
char file[50] ;

strcpy(file , pdbobj->file) ;
	
num_of_residues = pdbobj->numres ;

fh =  fopen(strcat(file, ".hessian") , "w") ;

for (n=1 ;  n<=3*num_of_residues ; n++)  {
	for (m=1 ;  m<=3*num_of_residues ; m++)  {
		fprintf (fh , "%10.5f " , hessian[n][m]) ;
	
	}
	fprintf (fh ,"\n") ;
}

fclose (fh) ;

}

void print_sparse_hessian(pdb *pdbobj , float **hessian , int natoms)  {
int m,n ;
int num_of_residues ;
FILE *fh ;	
char file[50] ;

strcpy(file , pdbobj->file) ;
	
num_of_residues = pdbobj->numres ;

fh =  fopen(strcat(file, ".sparsehessian") , "w") ;

for (n=1 ;  n<= 3*natoms ; n++)  {
	for (m=n ;  m<=3*natoms ; m++)  {
		if (hessian[n][m] != 0.0)  {
			fprintf (fh , "%8d%8d% 25.15e\n" , n , m , hessian[n][m]) ;
		}
	
	}
}

fclose (fh) ;

}

void print_kirchhof(pdb *pdbobj , float **kirchhof)  {
int m,n ;
int num_of_residues ;
FILE *fh ;	
char file[50] ;

strcpy(file , pdbobj->file) ;
	
num_of_residues = pdbobj->numres ;

fh =  fopen(strcat(file, ".kirchhof") , "w") ;

for (n=1 ;  n<=num_of_residues ; n++)  {
	for (m=1 ;  m<=num_of_residues ; m++)  {
		fprintf (fh , "%10.5f " , kirchhof[n][m]) ;
	
	}
	fprintf (fh ,"\n") ;
}

fclose (fh) ;

}


void print_pseudoinverse(pdb *pdbobj , float **invhessian)  {
int m,n ;
int num_of_residues ;
FILE *fh ;	
char file[50] ;

strcpy(file , pdbobj->file) ;
	
num_of_residues = pdbobj->numres ;

fh =  fopen(strcat(file, ".invhessian") , "w") ;

for (n=1 ;  n<=3*num_of_residues ; n++)  {
	for (m=1 ;  m<=3*num_of_residues ; m++)  {
		fprintf (fh , "%10.5f " , invhessian[n][m]) ;
	
	}
	fprintf (fh ,"\n") ;
}

fclose (fh) ;

}

void print_calb(pdb *pdbobj , float **invhessian) {
int m,n ;
int num_of_residues ;
FILE *fh ;	
char file[50] ;

strcpy(file , pdbobj->file) ;
	
num_of_residues = pdbobj->numres ;

fh =  fopen(strcat(file, ".calb") , "w") ;

for (n=1 ;  n<3*num_of_residues ; n=n+3)  {
	m = n/3 ;
	fprintf (fh , "%10.5f %10.5f\n" , invhessian[n][n]+ invhessian[n+1][n+1]+ invhessian[n+2][n+2] , pdbobj->residues[m].bfacca) ;
}

fclose (fh) ;

}




void print_indexing(pdb *pdbobj , double mean , double std)  {
int i , atmi ;
int num_of_residues ;
FILE *fh ;	
char file[50] ;

strcpy(file , pdbobj->file) ;
	
num_of_residues = pdbobj->numres ;

fh =  fopen(strcat(file, ".index") , "w") ;

for (i=0 ;  i<num_of_residues ; i++)  {
	for (atmi = 0 ; atmi <  pdbobj->residues[i].numberofatoms ; atmi++) 
		fprintf (fh , "%5d %3s %5d%c %c %3d %5s %8.3f %8.3f %8.3f %9.3f %9.3f\n" , pdbobj->residues[i].atoms[atmi].index , pdbobj->residues[i].resname , pdbobj->residues[i].pdbcoornum , 
	                  pdbobj->residues[i].insertioncode , pdbobj->residues[i].chain , pdbobj->residues[i].model , 
			  
			  pdbobj->residues[i].atoms[atmi].atomname , 
			  pdbobj->residues[i].atoms[atmi].coor[0] , pdbobj->residues[i].atoms[atmi].coor[1] , pdbobj->residues[i].atoms[atmi].coor[2] ,
			  
			  pdbobj->residues[i].atoms[atmi].bfac , (pdbobj->residues[i].atoms[atmi].bfac-mean)/std ) ; 
}

fclose (fh) ;

}


void print_log_file(char *file , char chain , int number_of_residues , float th) {

time_t t ;
FILE *fh ;	
char tmpfile[50] ;
char *ts ;

strcpy(tmpfile , file) ;

fh =  fopen(strcat(file, ".log") , "w") ;
t = time(NULL) ;
ts = ctime(&t) ;

fprintf (fh , "%s" , ts) ;
fprintf (fh , "file: %s chain: %c\n" , tmpfile , chain ) ;
fprintf (fh , "residues: %d\n" , number_of_residues ) ;
fprintf (fh , "th: %f\n" , th) ;

fclose (fh) ;

}

void pseudoinverse(float d[], float **v, float **crc, int n)  {

/**********************************************************************************************
// 
// Given the eigenvalues d[7..3*n] and eigenvectors v[7..n][1..n] 
// calculate the psedo inverse of the original symmetrical matrix.
// This inverse - crc corresponds to the correlation and the crosscorrelations as:
// <DrTi*Drj> = 3kbT/Gamma * crc(i,j)
//
// Eran (9/2005) 
//
**********************************************************************************************/


int k,j,i;

for (i=1;i<=3*n;i++)   {		 // index for row
	for (j=1;j<=3*n;j++)    {        // index for column
		crc[i][j] = 0.0 ;
		for (k=7;k<=3*n;k++)  {  // index for Eigenvalue and corresponding eigen vector igmoring the first
			crc[i][j] += (v[i][k]*v[j][k]/d[k]) ;
			// crc[j][i] = crc[i][j] ;
		}
	}
}

}


void calculate_bfac_stat(pdb *pdbobj , char chain , double *mean , double *std) {

/*************************************************************************************************
//
// Calculate Mean and STD to the B-factor values of the CA atoms
//
//
// Eran (10/2005)
//
*************************************************************************************************/

int i , atmi ;
int num_of_atoms=0 ;
int num_of_residues ;
double sum_sqdev=0.0 ;

*mean=0.0 ;

num_of_residues = pdbobj->numres ;

for (i=0 ; i<num_of_residues ; i++)  {
	if (((chain == pdbobj->residues[i].chain) || (chain == '*')) && (pdbobj->residues[i]. bfacca  != -9999))  {
        	for (atmi = 0 ; atmi <  pdbobj->residues[i].numberofatoms ; atmi++) { // loop over the atoms of residue i
			*mean += pdbobj->residues[i].atoms[atmi].bfac ;
			num_of_atoms++ ;
		}
	}
}

*mean /= num_of_atoms ;

for (i=0 ; i<num_of_residues ; i++)  {
	if (((chain == pdbobj->residues[i].chain) || (chain == '*')) && (pdbobj->residues[i]. bfacca  != -9999))  {
        	for (atmi = 0 ; atmi <  pdbobj->residues[i].numberofatoms ; atmi++) { // loop over the atoms of residue i
			sum_sqdev += (pdbobj->residues[i].atoms[atmi].bfac-(*mean))*(pdbobj->residues[i].atoms[atmi].bfac-(*mean)) ;
		}
	}
}

*std = sqrt(sum_sqdev/num_of_atoms) ;

}


/*

sparse matrix format:
01234567890123456789012345678901234567890
          1         2         3         4

       1       1    0.163988852795849E+02
       1       2    0.284541052833137E+01
       1       3   -0.224532282166162E+01
*/

